
#include <avr/io.h>
#include <util/delay.h>
#include "WConstants.h"
#include "global.h"
#include "mcp2515.h"
#include "mcp2515_defs.h"

#include "defaults.h"

// -------------------------------------------------------------------------
// Schreibt/liest ein Byte ueber den Hardware SPI Bus
//#define F_CAN	        				16000000UL							//MCP2515 OSC = 16MHz
#define F_CAN	        		20000000UL							//MCP2515 OSC = 20MHz

//BRP Setting in CNF1
#define CAN_1000kbps  			0       							// CAN speed at 1000 kbps
#define CAN_500kbps   			1       							// CAN speed at 500  kbps	
#define CAN_250kbps   			3   								// CAN speed at 250  kbps    	
#define CAN_125kbps   			7									// CAN speed at 125  kbps       	
#define CAN_62_5kbps   			9 									// CAN speed at 62.5 kbps

uint8_t spi_putc( uint8_t data )
{
  // put byte in send-buffer
  SPDR = data;
	
  // wait until byte was send
  while(!( SPSR & (1<<SPIF)));
  
  return SPDR;
}

// -------------------------------------------------------------------------
void mcp2515_write_register( uint8_t adress, uint8_t data )
{
  RESET(MCP2515_CS);
	
  spi_putc(SPI_WRITE);
  spi_putc(adress);
  spi_putc(data);
	
  SET(MCP2515_CS);
}

// -------------------------------------------------------------------------
uint8_t mcp2515_read_register(uint8_t adress)
{
  uint8_t data;
	
  RESET(MCP2515_CS);
	
  spi_putc(SPI_READ);
  spi_putc(adress);
  data = spi_putc(0xff);	
	
  SET(MCP2515_CS);
	
  return data;
}

// -------------------------------------------------------------------------
void mcp2515_bit_modify(uint8_t adress, uint8_t mask, uint8_t data)
{
  RESET(MCP2515_CS);
	
  spi_putc(SPI_BIT_MODIFY);
  spi_putc(adress);
  spi_putc(mask);
  spi_putc(data);
	
  SET(MCP2515_CS);
}

// ----------------------------------------------------------------------------
uint8_t mcp2515_read_status(uint8_t type)
{
  uint8_t data;
	
  RESET(MCP2515_CS);
	
  spi_putc(type);
  data = spi_putc(0xff);
	
  SET(MCP2515_CS);
	
  return data;
}

// -------------------------------------------------------------------------
uint8_t mcp2515_init(uint8_t speed)
{
  SET(MCP2515_CS);
  SET_OUTPUT(MCP2515_CS);
	
  RESET(P_SCK);
  RESET(P_MOSI);
  RESET(P_MISO);
	
  SET_OUTPUT(P_SCK);
  SET_OUTPUT(P_MOSI);
  SET_INPUT(P_MISO);
	
  // active SPI master interface
  SPCR = (1<<SPE)|(1<<MSTR) | (0<<SPR1)|(1<<SPR0);
  SPSR = 0;
	
  // reset MCP2515 by software reset.
  // After this he is in configuration mode.
  RESET(MCP2515_CS);
  spi_putc(SPI_RESET);
  SET(MCP2515_CS);
	
  // wait a little bit until the MCP2515 has restarted
  _delay_us(10);
	
  /*********************************************************************************
     *	CAN Bit Timings With Oscillator 20MHz
     *	Fosc 		  	= 20 MHz 
     *    Tosc 		  	= 1 / 20MHz = 50nS
     *    1TQ Minimum     = 2 Tosc 
     *
     *    1 Bit Time CAN = Sync Seg + Prop Seg + Phase Seg1(PS1) + Phase Seg2(PS2)
     *	Sync Seg   	= Fixed = 1TQ                       Best Quality = 1 TQ
     *	Prop Seg   	= (PRSEG0[0..7] + 1) * TQ = 1..8 TQ Best Quality = 3 TQ+
     *	Phase Seg1 	= (PHSEG1[0..7] + 1) * TQ = 1..8 TQ Best Quality = 3 TQ+
     *	Phase Seg2 	= (PHSEG2[0..7] + 1) * TQ = 1..8 TQ Best Quality = 3 TQ+
     *
     *    1TQ For 1 Bit Time  = [4..25]TQ Best Quality = 10 TQ(1+3+3+3)
     *    -> SJW:    Synchronization Jump Width Length = 1([0:0]+1)
     *    -> PRSEG:  Propagation Segment Length = 3([0:1:0]+1)
     *    -> PHSEG1: PS1 Length = 3([0:1:0]+1)
     *    -> PHSEG2: PS2 Length = 3([0:1:0]+1)
     *
     *    -> 62.5Kbps     = 1 / 62.5KHz = 16uS   
     *       Bit Time     = 16uS 
     *       Want 10TQ    = 16uS
     *       1TQ          = 16uS / 10 = 1.6uS
     *       1TQ:Tosc     = 1.6uS / 50ns
     *                    = 32 Tosc = 1600nS
     *       BRP          = (32 / 2)-1
    *                    = 15(0:0:1:1:1:1)   
     *
     *    -> 125Kbps      = 1 / 125KHz  = 8uS   
     *       Bit Time     = 8uS  
     *       Want 10TQ    = 8uS
     *       1TQ          = 8uS / 10 = 0.8uS
     *       1TQ:Tosc     = 0.8uS / 50ns
     *                    = 16 Tosc = 800nS
     *       BRP          = (16 / 2)-1
     *                    = 7(0:0:0:1:1:1)   
     *
     *    -> 250Kbps    	= 1 / 250KHz  = 4uS
     *       Bit Time  	= 4uS  
     *       Want 10TQ    = 4uS
     *       1TQ          = 4uS / 10 = 0.4uS
     *       1TQ:Tosc     = 0.4uS / 50ns
     *                    = 8 Tosc = 400nS
     *       BRP          = (8 / 2)-1
     *                    = 3(0:0:0:0:1:1)   
     *
     *    -> 500Kbps  	= 1 / 500KHz  = 2uS
     *       Bit Time 	= 2uS  
     *       Want 10TQ    = 2uS
     *       1TQ          = 2uS / 10=0.2uS
     *       1TQ:Tosc     = 0.2uS / 50ns
     *                    = 4 Tosc = 200nS
     *       BRP          = (4 / 2)-1
     *                    = 1(0:0:0:0:0:1)   
     *
     *    -> 1Mbps        = 1 / 1MHz = 1uS   
     *       Bit Time  	= 1uS  
     *       Want 10TQ    = 1uS
     *       1TQ          = 1uS / 10=0.1uS
     *       1TQ:Tosc     = 0.1uS / 50ns
     *                    = 2 Tosc = 100nS
     *       BRP          = (2 / 2)-1
     *                    = 0(0:0:0:0:0:0)   
     *********************************************************************************/
   //REQOP = 1:0:0(Set Configuration mode)
  mcp2515_bit_modify(CANCTRL,0xE0,(1<<REQOP2));
  
  //Configure Baud Rate Prescaler
  switch(speed)    
  {
    				   //Set to 62.5 kbps
                       //SJW=00(Length=1 TQ), BRP = 15(TQ = 2 x (15+1)/FOSC)
                       //CNF1 = 0:0 + 0:0:1:1:1:1 -> 62.5 Kbps
    case CAN_62_5kbps: mcp2515_write_register(CNF1,(1<<BRP3)|(1<<BRP2)|(1<<BRP1)|(1<<BRP0)); 
                       break;

					   //Set to 125 kbps
                       //SJW=00(Length=1 TQ), BRP = 7(TQ = 2 x (7+1)/FOSC)
                       //CNF1 = 0:0 + 0:0:0:1:1:1 -> 125  Kbps
    case CAN_125kbps:  mcp2515_write_register(CNF1,(1<<BRP2)|(1<<BRP1)|(1<<BRP0));        	                   
                       break;

                       //Set to 250 kbps
                       //SJW=00(Length=1 TQ), BRP = 3(TQ = 2 x (3+1)/FOSC)
					   //CNF1 = 0:0 + 0:0:0:0:1:1 -> 250  Kbps
    case CAN_250kbps:  mcp2515_write_register(CNF1,(1<<BRP1)|(1<<BRP0));         
                       break;

                       //Set to 500 kbps
					   //SJW=00(Length=1 TQ), BRP = 1(TQ = 2 x (1+1)/FOSC) 
                       //CNF1 = 0:0 + 0:0:0:0:0:1 -> 500  Kbps
    case CAN_500kbps:  mcp2515_write_register(CNF1,(1<<BRP0));        
                       break;

                       //Set to 1000 kbps
                       //SJW=00(Length=1 TQ), BRP = 0(TQ = 2 x (0+1)/FOSC)
                       //CNF1 = 0:0 + 0:0:0:0:0:0 -> 1000 Kbps
    case CAN_1000kbps: mcp2515_write_register(CNF1,0);    		
                       break;

                       //unrecognized or null command
    default:           break;         
  }
	 
  #if(F_CAN == 20000000UL)
  //If OSC = 20MHz 
  //BTLMODE=1,PHSEG1=3TQ(0:1:0),PRSEG=3TQ(0:1:0)
  //CNF2 = 1+0+0:1:0+0:1:0 
  mcp2515_write_register(CNF2,(1<<BTLMODE)|(1<<PHSEG11)|(1<<PRSEG1));
  
  //PHSEG2=3TQ(0:1:0)
  //CNF3 = 0+0+x:x:x:0:1:0 
  mcp2515_write_register(CNF3,(1<<PHSEG21));
  #endif
  
  #if(F_CAN == 16000000UL)  
  //If OSC = 16MHz 
  //BTLMODE=1,PHSEG1=3TQ(0:1:0),PRSEG=2TQ(0:0:1)
  //CNF2 = 1+0+0:1:0+0:0:1 
  mcp2515_write_register(CNF2,(1<<BTLMODE)|(1<<PHSEG11)|(1<<PRSEG0));

  //PHSEG2=2TQ(0:0:1)
  //CNF3 = 0+0+x:x:x:0:0:1 
  mcp2515_write_register(CNF3,(1<<PHSEG20));
  #endif
  /* End of CAN Baudrate Generate Config */
	
  // test if we could read back the value => is the chip accessible?
  if (mcp2515_read_register(CNF1) != speed) 
  {
	return false;
  }

  //Enable RX1,RX0 all Interrupts 
  mcp2515_write_register(CANINTE,(1<<RX1IE)|(1<<RX0IE)); 
		
  // Buffer 0 : Receive All Message
  mcp2515_write_register(RXB0CTRL,(1<<RXM1)|(1<<RXM0));
	
  //Buffer 1 : Receive All Message
  mcp2515_write_register(RXB1CTRL,(1<<RXM1)|(1<<RXM0));

  //Clear All Data Register	
  uint8_t temp[4] = { 0, 0, 0, 0 };
	
  //Enable RX1BF,RX0BF Output Pins
  mcp2515_write_register(BFPCTRL,(1<<B1BFE)|(1<<B0BFE)|(1<<B1BFM)|(1<<B0BFM));
	
  //Disable TX0RTS,TX1RTS,TX2RTS Input Pin
  mcp2515_write_register( TXRTSCTRL, 0);
	
  //REQOP = 0:0:0 (Set Normal Operation mode)
  mcp2515_bit_modify(CANCTRL, 0xE0, 0);

  //Exit
  return true;
}

// ----------------------------------------------------------------------------
uint8_t mcp2515_check_message(void) 
{
  uint8_t temp;

  temp = mcp2515_read_register(CANINTF);

  // Verify RX0,RX1 Frame Receive
  if ( temp & ((1<<RX0IF)|(1<<RX1IF)) ) 
  {
    return true;
  }
  else
  {
    return false;
  }
}

// ----------------------------------------------------------------------------
uint8_t mcp2515_check_free_buffer(void)
{
  uint8_t status = mcp2515_read_status(SPI_READ_STATUS);
  if ((status & 0x54) == 0x54) 
  {
	// all buffers used
	return false;
  }
  return true;
}

// ----------------------------------------------------------------------------
uint8_t mcp2515_get_message(tCAN *message)
{
  // read status
  uint8_t status = mcp2515_read_status(SPI_RX_STATUS);
  uint8_t addr;
  uint8_t t;
  if (bit_is_set(status,6)) 
  {
	// message in buffer 0
	addr = SPI_READ_RX;
  }
  else if (bit_is_set(status,7)) 
  {
	// message in buffer 1
	addr = SPI_READ_RX | 0x04;
  }
  else 
  {
	return 0;
  }

  RESET(MCP2515_CS);
  spi_putc(addr);
	
  // read id
  message->id  = (uint16_t) spi_putc(0xff) << 3;
  message->id |=            spi_putc(0xff) >> 5;
	
  spi_putc(0xff);
  spi_putc(0xff);
	
  // read DLC
  uint8_t length = spi_putc(0xff) & 0x0f;
	
  message->header.length = length;
  message->header.rtr = (bit_is_set(status, 3)) ? 1 : 0;
	
  // read data
  for (t=0;t<length;t++) 
  {
	message->data[t] = spi_putc(0xff);
  }
  SET(MCP2515_CS);
	
  // clear interrupt flag
  if (bit_is_set(status, 6)) 
  {
	mcp2515_bit_modify(CANINTF, (1<<RX0IF), 0);
  }
  else 
  {
	mcp2515_bit_modify(CANINTF, (1<<RX1IF), 0);
  }
	
  return (status & 0x07) + 1;
}

// ----------------------------------------------------------------------------
uint8_t mcp2515_send_message(tCAN *message)
{
	uint8_t status = mcp2515_read_status(SPI_READ_STATUS);
	
	/* Statusbyte:
	 *
	 * Bit	Function
	 *  2	TXB0CNTRL.TXREQ
	 *  4	TXB1CNTRL.TXREQ
	 *  6	TXB2CNTRL.TXREQ
	 */
	uint8_t address;
	uint8_t t;
	
	if (bit_is_clear(status, 2)) 
	{
	  address = 0x00;
	}
	else if (bit_is_clear(status, 4)) 
	{
	  address = 0x02;
	} 
	else if (bit_is_clear(status, 6)) 
	{
	  address = 0x04;
	}
	else 
	{
	  // all buffer used => could not send message
	  return 0;
	}
	
	RESET(MCP2515_CS);
	spi_putc(SPI_WRITE_TX | address);
	
	spi_putc(message->id >> 3);
    spi_putc(message->id << 5);
	
	spi_putc(0);
	spi_putc(0);
	
	uint8_t length = message->header.length & 0x0f;
	
	if (message->header.rtr) 
	{
	  // a rtr-frame has a length, but contains no data
	  spi_putc((1<<RTR) | length);
	}
	else 
	{
	  // set message length
	  spi_putc(length);
		
	  // data
	  for (t=0;t<length;t++) 
	  {
		spi_putc(message->data[t]);
	  }
	}
	SET(MCP2515_CS);
	
	_delay_us(1);
	
	// send message
	RESET(MCP2515_CS);
	address = (address == 0) ? 1 : address;
	spi_putc(SPI_RTS | address);
	SET(MCP2515_CS);
	
	return address;
}
